/*
 *   This file is part of the MLV Library.
 *
 *   Copyright (C) 2010 Adrien Boussicault, Marc Zipstein
 *
 *
 *    This Library is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This Library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this Library.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "MLV_sound.h"

#include "sound.h"

#include "warning_error.h"

#include "memory_management.h"

#include "data_structure.h"

#include <glib.h>

extern DataMLV* MLV_data;

struct _MLV_Sheet_music {
	SDL_AudioSpec wav_spec;
	Uint8* wav_buffer;
	Uint32 wav_length;
};

struct _MLV_Playlist {
    GList* list_music;
};

void close_stream( MLV_Stream* stream, void* useless ){
	MLV_data->playing_streaming = g_list_remove( MLV_data->playing_streaming, stream );
	MLV_FREE( stream, MLV_Stream );
}

void audio_callback(void *userdata, Uint8 *stream, int len){
    GList* finished_streaming = NULL;
    SDL_LockAudio();
	if( SDL_SemWait( MLV_data->audio_semaphore )  ){
		ERROR("lockPrompt - Probleme de semaphore");
	}
	GList* tmp = MLV_data->playing_streaming;
	while(tmp){
		MLV_Stream* stream_data = tmp->data;
		if ( stream_data->audio_len != 0 ){
		/* Mix as much data as possible */
			len = ( len > stream_data->audio_len ? stream_data->audio_len : len );
			SDL_MixAudio(stream, stream_data->audio_pos, len, SDL_MIX_MAXVOLUME);
			stream_data->audio_pos += len;
			stream_data->audio_len -= len;
		}else{
			finished_streaming = g_list_prepend(
				finished_streaming,
				stream_data	
			);
		}
		tmp = tmp->next;
	}
	g_list_foreach(
		finished_streaming,
		(void (*)(void*,void*)) close_stream
		, NULL
	);
    g_list_free( finished_streaming );

	if( SDL_SemPost( MLV_data->audio_semaphore ) ){
		ERROR("unlockPrompt - Probleme de semaphore");
	}
    SDL_UnlockAudio();
}

void MLV_init_sound(
	int freq,
	Uint16 format
){
    MLV_data->audio_semaphore = SDL_CreateSemaphore(1);
	MLV_data->playing_streaming = NULL;

	SDL_AudioSpec wav_spec;
    wav_spec.freq = freq;
    wav_spec.format = format;
    wav_spec.samples = 4096;
	wav_spec.userdata=NULL;
    wav_spec.callback = audio_callback;

	wav_spec.channels = 2;
	wav_spec.silence = 0;
	wav_spec.size = 0;
	wav_spec.userdata = 0;

	if ( SDL_OpenAudio(&(wav_spec), &(MLV_data->audio_device_spec) ) < 0 ){
		fprintf(stderr, "Couldn't open audio: %s - line: %i, file: %s\n", SDL_GetError(), __LINE__, __FILE__);
		exit(-1);
	}
    SDL_PauseAudio(0);
}

void MLV_close_sound(){
    SDL_PauseAudio(1);
	SDL_CloseAudio();
	MLV_stop_all_music();
    SDL_DestroySemaphore( MLV_data->audio_semaphore );
}

MLV_Sheet_music* MLV_load_music( const char* file_music ){
	MLV_Sheet_music* music = MLV_MALLOC( 1, MLV_Sheet_music );
//    if( SDL_LoadWAV( file_music, &(music->wav_spec), &(music->wav_buffer), &(music->wav_length) ) == NULL ){
    if( SDL_LoadWAV( file_music, &(music->wav_spec), &(music->wav_buffer), &(music->wav_length) ) == NULL ){
      fprintf(stderr, "Could not open %s : (SDL ERROR : %s ) - line : %i, file : %s \n", file_music, SDL_GetError(), __LINE__, __FILE__);
      exit(-1);
    }
	return music;
}

void register_stream( MLV_Stream* stream ){
    if( SDL_SemWait( MLV_data->audio_semaphore )  ){
        ERROR("lockPrompt - Probleme de semaphore");
    }
    MLV_data->playing_streaming = g_list_prepend(
		MLV_data->playing_streaming ,
		stream
	);
    if( SDL_SemPost( MLV_data->audio_semaphore ) ){
        ERROR("unlockPrompt - Probleme de semaphore");
    }
}

const MLV_Stream* open_music( MLV_Sheet_music* music, float volume ){
	MLV_Stream* stream = MLV_MALLOC( 1, MLV_Stream );
	stream->music = music;
	stream->audio_len = music->wav_length;
	stream->audio_pos = music->wav_buffer;
	register_stream( stream );
	return stream;
}

void MLV_play_music( MLV_Sheet_music* music, float volume ){
	open_music( music, volume );
}

void MLV_close_music( MLV_Sheet_music* music ){
	SDL_FreeWAV( music->wav_buffer );
	MLV_FREE( music, MLV_Sheet_music );
}

void MLV_stop_all_music(){
	if( SDL_SemWait( MLV_data->audio_semaphore )  ){
		ERROR("lockPrompt - Probleme de semaphore");
	}
    GList* streams = NULL;
	GList* tmp = MLV_data->playing_streaming;
	while(tmp){
		streams = g_list_prepend(
			streams,
			tmp->data	
		);
		tmp = tmp->next;
	}
	g_list_foreach(
		streams,
		(void (*)(void*,void*)) close_stream
		, NULL
	);
    g_list_free( streams );
    if( SDL_SemPost( MLV_data->audio_semaphore ) ){
        ERROR("unlockPrompt - Probleme de semaphore");
    }
}

