/*
 *   This file is part of the MLV Library.
 *
 *   Copyright (C) 2010 Adrien Boussicault, Marc Zipstein
 *
 *
 *    This Library is free software: you can redistribute it and/or modify
 *    it under the terms of the GNU General Public License as published by
 *    the Free Software Foundation, either version 3 of the License, or
 *    (at your option) any later version.
 *
 *    This Library is distributed in the hope that it will be useful,
 *    but WITHOUT ANY WARRANTY; without even the implied warranty of
 *    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *    GNU General Public License for more details.
 *
 *    You should have received a copy of the GNU General Public License
 *    along with this Library.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * \file MLV_text.h
 *
 * \author Adrien Boussicault
 * \author Marc Zipstein
 *
 * \brief Ce fichier définit les différents prototypes des fonctions permmettant d'afficher du texte et des boîtes de dialogues.
 */

#ifndef __MLV_TEXTBOX_H__
#define __MLV_TEXTBOX_H__

#include "MLV_color.h"

/**
 * \brief Les différents types de justification du texte.
 */
typedef enum {
	MLV_TEXT_CENTER, /**< Le texte est justifié au centre. */
	MLV_TEXT_LEFT, /**< Le texte est justifié sur la gauche. */
	MLV_TEXT_RIGHT /**< Le texte est justifié sur la doite. */
} MLV_Text_justification;

/**
 * \brief Les différents types de position horizontale du texte dans une boîte.
 */
typedef enum {
	MLV_HORIZONTAL_CENTER, /**< Le texte est centrée horizontalement dans la boîte. */
	MLV_HORIZONTAL_LEFT, /**< Le texte est positioné sur la droite de la boîte. */
	MLV_HORIZONTAL_RIGHT /**< Le texte est positioné sur la gauche de la boîte. */
} MLV_Horizontal_position;

/**
 * \brief Les différents types de positions verticales du texte dans une boîte.
 */
typedef enum {
	MLV_VERTICAL_CENTER, /**< Le texte est centré horizonatalement dans la boîte. */
	MLV_VERTICAL_TOP, /**< Le texte est positionné en haut de la boîte. */
	MLV_VERTICAL_BOTTOM /**< Le texte est positionné en bas de la boîte. */
} MLV_Vertical_position;

/**
 * Type des polices de caractères dans la librairie MLV.
 */
typedef struct _MLV_Font MLV_Font;

/**
 * \brief Charge en mémoire une police de caractères de type True Type .
 *
 * La police chargée est configurée avec une taille de police donnée en paramètre.
 *
 * \param file_font Le chemin d'accès où se trouve le fichier contenant la police de caractères.
 * \param size La taille de la police
 * \return La structure de donée MLV contenant la police de caractères. 
 */
MLV_Font* MLV_load_font( const char* file_font, int size );

/**
 * \brief Libère la mémoire d'une police de caractères précédement chargée.
 * \param font L'adresse de la police de caractères à libérer.
 */
void MLV_close_font( MLV_Font* font );

/**
 * \brief Calcule la taille du texte qui sera affiché sur l'écran à l'aide de la fonction MLV_text.
 *
 * \param text Texte qui doit être affiché
 * \param width Largeur du texte
 * \param height Hauteur du texte
 */
void MLV_text_size( const char *text, int *width, int *height );

/**
 * \brief Calcule la taille du texte qui sera affiché sur l'écran à l'aide de la fonction MLV_text_with_font.
 *
 * \param text Texte qui doit être affiché

 * \param width Largeur du texte
 * \param height Hauteur du texte
 * \param font La police de caractères utilisée pour afficher le texte.
 */
void MLV_text_size_with_font( const char *text, int *width, int *height, MLV_Font* font );

/**
 * \brief Imprime un texte donné à une position et une couleur données.
 * 
 * Le texte est imprimé à l'aide de la police de caractères par default de la librairie MLV.
 *
 * \param x Coordonnée en X du coin Nord-Ouest du texte
 * \param y Coordonnée en Y du coin Nord-Ouest du texte
 * \param text texte à afficher
 * \param color couleur du tracé
 */
void MLV_draw_text(int x, int y, const char *text, MLV_Color color);

/**
 * \brief Écrit du texte su l'écran en utilisant une couleur et une police de caractères données en paramètres.
 * 
 * \param x Coordonnée en X du coin Nord-Ouest du texte
 * \param y Coordonnée en Y du coin Nord-Ouest du texte
 * \param text texte à afficher
 * \param color couleur du tracé
 * \param font La police de caractères utilisée pour dessiner le texte.
 */
void MLV_draw_text_with_font(int x, int y, const char *text, MLV_Font* font, MLV_Color color);

/**
 * \brief Dessine une boîte contenant du texte.
 *
 * La taille de la boîte dessinée s'adapte à la
 * taille du texte qui est situé à l'interieur.
 * Les différentes couleurs de la boîte et du texte,
 * ainsi que la taille de l'interligne du texte sont
 * paramétrables.
 *
 * Il est possible de préciser la justiifcation du
 * texte à l'aide du champs : text_justification
 *                             
 * Enfin, la fonction renvoie la taille de la boîte 
 * grâce aux champs result_width et result_height.
 * L'utilisateur peut passer des pointeurs NULL à 
 * ces deux champs si il ne désire pas récupérer la 
 * taille de la boîte créé. 
 *
 * \param x Coordonnée en X de la position du coin Nord-Ouest de la boîte
 * \param y Coordonnée en X de la position du coin Nord-Ouest de la boîte
 * \param message Texte à afficher
 * \param borderColor Couleur des bordures
 * \param sizeInterligne Taille de l'interligne
 * \param textColor Couleur du texte
 * \param backgroundColor Couleur du fond de la boîte
 * \param text_justification Justification du texte.
 * \param result_width adresse de la mémoire dans laquelle la largeur de la fenêtre dessinée sera enregistrée.
 * \param result_height adresse de la mémoire dans laquelle la hauteur de la fenêtre dessinée sera enregistrée.
 */
void MLV_draw_adapted_text_box(
	int x, int y,
	const char* message,
	int sizeInterligne,
	MLV_Color borderColor, MLV_Color textColor, MLV_Color backgroundColor,
	MLV_Text_justification text_justification,
	int *result_width, int *result_height
);

/**
 * \brief Calcul la taille de la boîte de texte qui serait obtenue si vous utilisez la fonction : MLV_adapted_text_box.
 *
 * \param message texte qui sera affiché lorsque vous utiliserez MLV_adapted_box.
 * \param sizeInterligne Taille de l'interligne
 * \param result_width Largeur de la boîte que vous obtiendrez en utilisant MLV_adapted_box.
 * \param result_height Hauteur de la boîte que vous obtiendrez en utilisant MLV_adapted_box.
 */
int MLV_adapted_text_box_size(
	const char* message,
	int sizeInterligne,
	int *result_width, int *result_height
);

/**
 * \brief Même chose que MLV_draw_adapted_text_box mis à part que le texte est rendu avec une police de caractères donnée en paramètre.
 *
 * \param x Coordonnée en X de la position du coin Nord-Ouest de la boîte
 * \param y Coordonnée en X de la position du coin Nord-Ouest de la boîte
 * \param message Texte à afficher
 * \param font La police de caractères utilisée pour dessiner le texte.
 * \param borderColor Couleur des bordures
 * \param sizeInterligne Taille de l'interligne
 * \param textColor Couleur du texte
 * \param backgroundColor Couleur du fond de la boîte
 * \param text_justification Justification du texte.
 * \param result_width adresse de la mémoire dans laquelle la largeur de la fenêtre dessinée sera enregistrée.
 * \param result_height adresse de la mémoire dans laquelle la hauteur de la fenêtre dessinée sera enregistrée.
 */
void MLV_draw_adapted_text_box_with_font(
	int x, int y,
	const char* message,
	MLV_Font* font, 
	int sizeInterligne,
	MLV_Color borderColor, MLV_Color textColor, MLV_Color backgroundColor,
	MLV_Text_justification text_justification,
	int *result_width, int *result_height
);

/**
 * \brief Calcul la taille de la boîte de texte qui serait obtenue si vous utilisez la fonction : MLV_adapted_text_box_with_font.
 *
 * \param message texte qui sera affiché lorsque vous utiliserez MLV_adapted_box.
 * \param font La police de caractères utilisée pour dessiner le texte.
 * \param sizeInterligne Taille de l'interligne
 * \param result_width Largeur de la boîte que vous obtiendrez en utilisant MLV_adapted_box.
 * \param result_height Hauteur de la boîte que vous obtiendrez en utilisant MLV_adapted_box.
 */
int MLV_draw_adapted_text_box_size_with_font(
	const char* message,
	MLV_Font* font, 
	int sizeInterligne,
	int *result_width, int *result_height
);

/**

 * \brief Dessine une boîte de taille donnée par l'utilisateur et contenant un texte donné par l'utilisateur.
 *
 * Le texte est déssiné à l'interieur d'une boîte.
 * Le texte est imprimé sur plusieurs lignes en respectant les retours à la ligne 
 * présent dans le texte.
 *
 * Les différentes couleurs de la boîte et du texte,
 * ainsi que la taille de l'interligne du texte sont
 * paramétrables.
 *
 * Il est possible de préciser la justification du
 * texte contenu dans la boîte.
 *
 * De même , il est possible de préciser la position du 
 * texte à l'intérieur de la boîte.
 *                     
 * Cette fonction ne remet pas en forme le texte. Cela 
 * veux dire que si le texte trop grand alors il depassera 
 * de la boîte et les parties qui depassent ne seront pas 
 * affichées.
 *
 * \param x Coordonnée en X de la position du coin Nord-Ouest de la boîte
 * \param y Coordonnée en Y de la position du coin Nord-Ouest de la boîte
 * \param width hauteur de la boîte
 * \param height largeur de la boîte
 * \param message texte à afficher
 * \param sizeInterligne taille de l'interligne
 * \param borderColor Couleur des bordures
 * \param textColor Couleur du texte
 * \param backgroundColor Couleur du fond de la boîte.
 * \param text_justification Justification du texte.
 * \param horizontal_position Position horizontale du texte dans la boîte.
 * \param vertical_position Position verticale du texte dans la boîte.
 */
void MLV_draw_text_box( 
	int x, int y, 
	int width, int height, 
	const char* message , 
	int sizeInterligne,
	MLV_Color borderColor, MLV_Color textColor, MLV_Color backgroundColor,
	MLV_Text_justification text_justification,
	MLV_Horizontal_position horizontal_position,
	MLV_Vertical_position vertical_position
);

/**
 * \brief Même chose que la fonction MLV_draw_text_box mis à part que le texte est rendu avec une police de caractères donnée en paramètre par l'utilisateur.
 * 
 * \param x Coordonnée en X de la position du coin Nord-Ouest de la boîte
 * \param y Coordonnée en Y de la position du coin Nord-Ouest de la boîte
 * \param width hauteur de la boîte
 * \param height largeur de la boîte
 * \param message texte à afficher
 * \param font La police de caractères utilisée pour dessiner le texte.
 * \param sizeInterligne taille de l'interligne
 * \param borderColor Couleur des bordures
 * \param textColor Couleur du texte
 * \param backgroundColor Couleur du fond de la boîte.
 * \param text_justification Justification du texte.
 * \param horizontal_position Position horizontale du texte dans la boîte.
 * \param vertical_position Position verticale du texte dans la boîte.
 */
void MLV_draw_text_box_with_font( 
	int x, int y, 
	int width, int height, 
	const char* message , 
	MLV_Font* font, 
	int sizeInterligne,
	MLV_Color borderColor, MLV_Color textColor, MLV_Color backgroundColor,
	MLV_Text_justification text_justification,
	MLV_Horizontal_position horizontal_position,
	MLV_Vertical_position vertical_position
);
#endif

